;; message broker

;; Manages a list of device-ids who are allowed to send and receive
;; data.
;;
;; A device state consists of a timestamp, temperature and led
;; state.
;;
;; Each device can subsribe to a topic (which is simply a string, it
;; consists of the sender's id plus "/" plus a string), there is a
;; simple ruleset: Every subscribed device gets the new state.
;;
;; Each device can ask for the "shadow device" (i.e. the last sent
;; devices' state).
;;
;; No authentication is done. Everything can mess up everything.

#lang racket/base

(require "common.rkt"
         2htdp/image
         2htdp/universe)

;; constant for my id
(define BROKER-ID "broker")

;; registry is a hash table with key equals the device id, value is
;; the iworld? if a world has connected #f otherwise. 

;; checks if a given parameter is a registry
;; any? -> boolean? 
(define (registry? reg)
  (hash? reg))

;; checks if a given string is a registered device
;; string? -> boolean? 
(define (registered-device? reg device-id)
  (and (registry? reg)
       (hash-has-key? reg device-id)))

;; Gives device's world if any (#f otherwise
(define (world-of-device reg device-id)
  (hash-ref reg device-id #f))

;; registers a new device with iworld? or #f as value
(define (register-device reg device-id wrld)
  (if (and (registry? reg)
           (string? device-id)
           (or (iworld? wrld)
               (equal? #f wrld)))
      (hash-set reg device-id wrld)
      (error (format "arguments (~a,~a,~a) are not registry? string? (or/c iworld? #f) (~a,~a,~a)"
                     reg device-id wrld (registry? reg) (string? device-id) (or (iworld? wrld)
                                                                                (equal? wrld #f))))))

;; shadows is a hash table
;; - key is the device id (string?)
;; - values is the device's last state (state?) ("Schattengeräte" at Amazon IoT)

;; checks if shadows are valid
;; any? -> boolean?
(define (shadows? shadows)
  (hash? shadows))

;; returns a shadow device's state if in the shadow's hash table, #f
;; otherwise
;; string? shadows? -> (or/c device? #f)
(define (shadow-device-ref shadows device-id)
  (if (shadows? shadows)
      (hash-ref shadows device-id #f)
      #f))

;; sets the shadow device's state in the hash table
;; string? device? -> hash?
(define (shadow-device-set shadows device-id dev)
  (if (and (shadows? shadows) (string? device-id) (device? dev))
      (hash-set shadows device-id dev)
      (error (format "arguments (~a,~a,~a) are not shadows? string? device? (~a,~a,~a)"
                     shadows device-id dev
                     (shadows? shadows? (string? device-id) (device? dev))))))

;; subscriptions are a list of cons cells, where car is the topic
;; to subscribe to and the cdr is the recipient that want to get the
;; devie's state
;; (("abc" . "def") ( ... ))

;; checks if any? is a bunch of subscriptions
;; any? -> boolean?
(define (subscriptions? subs)
  (andmap pair? subs))

;; Gives all subscribers for a topic
;; subscriptions? string? -> ListOf string?
(define (subscribers subs topic)
  (if (and (subscriptions? subs)
           (string? topic))
      (map cdr (filter (lambda (ele)
                         (string=? topic (car ele))) subs))
      (error (format "arguments (~a,~a) are not subscriptions? string? (~a,~a)"
                     subs topic (subscriptions? subs) (string? topic)))))

;; If a list of strings is given it creates for each id a pair with
;; the recipient; if only a string is given, just return a list with
;; one pair. 
;; (or/c (ListOf string?) string?)  -> listof pair?
(define (subscribe-ids rcpt ids)
  (cond
    [(string? ids)
     (list (cons ids rcpt))]
    [(and (list? ids) (andmap string? ids))
     (map (lambda (id) (cons id rcpt)) ids)]))

;; XXX ... 

;; UniverseState is a struct of
;; - registry (list?) 
;; - shadow devices (hash?) 
;; - subscriptions (list?) 
;; - list of worlds (list?

(struct broker (registry shadows subscriptions worlds) #:transparent)

;; gives an initial empty broker with a list of registered devices
;;  -> broker?
(define (empty-broker)
  (broker (hash) (hash) (list) (list)))

;; checks if a broker is valid
;; broker? -> boolean? 
(define (valid-broker? brk)
  (and (broker? brk)
       (registry? (broker-registry brk))
       (shadows? (broker-shadows brk))
       (subscriptions? (broker-subscriptions brk))
       (list? (broker-worlds  brk))))

;; adds a new component to the broker and sends the MSG-ID to the new
;; device. A Raspberry Pi should respond with just his id (MSG-ID), a
;; monitor should answer with the topic he wishes to subscribe to (MSG-SUBSCRIBE).
;; UniverseState iworld? -> (or/c UniverseState bundle?)
(define (new-member us new-world)
  ;; XXX: Check if it is a member of the registered devices
  (make-bundle (struct-copy broker us
                            [worlds (cons new-world (broker-worlds us))])
               (list (make-mail new-world (message BROKER-ID MSG-ID #f)))
               '()))

;; somebody talks to us
;; UniverseState iworld? sexp? -> (or/c UniverseState bundle?)
(define (new-msg us wrld msg)
  (if (and (message? msg) (valid-message? msg))
      (cond
        ;; Someone sends us his id: Just add to the registry
        [(and (equal? (message-type msg) MSG-ID)
              (string? (message-body msg)))
         (struct-copy broker us
                      [registry (register-device (broker-registry us)
                                                 (message-sender msg)
                                                 wrld)])]
        ;; Someone wants to subscribe: Just subscribe and no answer;
        ;; If only a string is given: subscribe to this id
        ;; If a list of strings is given: subsribe to all ids
        [(and (equal? (message-type msg) MSG-SUBSCRIBE)
              (or (string? (message-body msg))
                  (list? (message-body msg))))
         (struct-copy broker us
                      [registry (register-device (broker-registry us)
                                                 (message-sender msg)
                                                 wrld)]
                      [subscriptions (append
                                      (subscribe-ids wrld (message-body msg))
                                      (broker-subscriptions us))])]
        ;; Somebody publishes data: Send this data to all subscribed worlds;
        ;; update shadow device 
        [(and (equal? (message-type msg) MSG-PUBLISH)
              (device? (message-body msg))
              (valid-device-state? (message-body msg)))
         (make-bundle (struct-copy broker us
                                   [shadows (shadow-device-set (broker-shadows us)
                                                               (message-sender msg)
                                                               (message-body msg))])
                      (map (lambda (rcpt) (make-mail rcpt msg)) (subscribers
                                                                 (broker-subscriptions us)
                                                                 (message-sender msg)))
                      '())]
        [(and (equal? (message-type msg) MSG-CHANGE-STATE)
              (change-device-state? (message-body msg))
              (valid-change-device-state? (message-body msg)))
         (make-bundle us
                      (list (make-mail (world-of-device
                                        (broker-registry us)
                                        (change-device-state-device
                                         (message-body msg)))
                                       (message  BROKER-ID
                                                 MSG-CHANGE-STATE
                                                 (message-body msg))))
                      '())]
        [else us])
      us))

(define (main)
  (universe (empty-broker)
            (state #t)
            (on-new new-member)
            (on-msg new-msg)))

(module+ main
  (main))

(module+ test
  (require rackunit)

  (check-true (registry? (hash)))
  (check-true (registry? (hash "a" "x" "b" "y")))
  (check-false (registry? '("a" "b" 3)))
  (check-false (registry? '("a" "b" #t)))

  (define REGISTRY (hash "pi1" #f "monitor1" #f))
  
  (check-true (registered-device? (hash  "pi_000001" #f) "pi_000001" ))
  (check-false (registered-device? (hash) "abc" ))
  (check-true (registered-device? REGISTRY "pi1"))
  (check-false (world-of-device REGISTRY "pi1"))
  (check-equal? (world-of-device (hash "pi1" iworld1) "pi1") iworld1)
  
  (check-true (shadows? (hash "abc" "123")))
  (check-false (shadows? (list "abc" "345")))

  (define DEV1 (device 14773198700 18 #f))
  (define DEV3 (device 14773198800 4 #t))
  
  (define SHADOWS (hash "pi1" (device 1477319668 20 #f)
                        "pi2" (device 1477319670 25 #t)))

  (check-true (shadows? SHADOWS))
  (check-true (device? (shadow-device-ref SHADOWS "pi1")))
  (check-false (device? (shadow-device-ref SHADOWS "abc")))

  (check-exn exn:fail? (lambda () (shadow-device-set #t "pi1" DEV1)))
  (check-exn exn:fail? (lambda () (shadow-device-set (hash) 123 DEV1)))
  (check-exn exn:fail? (lambda () (shadow-device-set (hash) "pi1" "abc")))

  (check-equal? (shadow-device-set SHADOWS "pi3" DEV3)
                '#hash(("pi1" . #s(device 1477319668 20 #f))
                       ("pi2" . #s(device 1477319670 25 #t))
                       ("pi3" . #s(device 14773198800 4 #t))))
  (check-equal? (shadow-device-set SHADOWS "pi1" DEV1)
                '#hash(("pi1" . #s(device 14773198700 18 #f))
                       ("pi2" . #s(device 1477319670 25 #t))))

  (check-true (subscriptions? (list)))
  (check-true (subscriptions? (list (cons 1 2) (cons 3 4))))
  (check-exn exn:fail? (lambda () (subscriptions? #f)))
  (check-false (subscriptions? (list 1 2 3)))

  (check-equal? (subscribe-ids "foo" '("1" "2" "3"))
                '(("1" . "foo") ("2" . "foo") ("3" . "foo")))

  (check-equal? (subscribe-ids "foo" '("1"))
                '(("1" . "foo")))
  (check-equal? (append (subscribe-ids "foo" '("1")) (subscribe-ids "bar" '("2")))
                '(("1" . "foo") ("2" . "bar")))

  (check-equal? (subscribe-ids "foo" "1") '(("1" . "foo")))
  
  (check-exn exn:fail? (lambda () (subscribers #t #t)))
  (check-exn exn:fail? (lambda () (subscribers (list) #t)))

  (define SUBS (list (cons "1" "a") (cons "2" "b") (cons "1" "c")))
  
  (check-equal? (subscribers SUBS "1") '("a" "c"))
  (check-equal? (subscribers SUBS "2") '("b"))
  (check-equal? (subscribers SUBS "3") '())

  (check-equal? (empty-broker)  (broker (hash) (hash) (list) (list)))

  (define BROKER0 (broker REGISTRY (hash) (list) (list)))
  
  (check-true (valid-broker? BROKER0))
  (check-false (valid-broker? (broker #t #t #t #t)))

  (check-equal? (new-member BROKER0 iworld1)
                (make-bundle
                 (broker (hash "pi1" #f "monitor1" #f)
                         (hash) (list) (list iworld1))
                 (list (make-mail iworld1 (message BROKER-ID MSG-ID #f)))
                 '()))

  (define MSG1 (message "monitor1" MSG-SUBSCRIBE "pi1"))
  (define MSG2 (message "pi1" MSG-PUBLISH
                        (device 1477391377 17 #t)))
  (define MSG3 (message "pi1" MSG-ID "pi1"))
  (define MSG4 (message "monitor1" MSG-CHANGE-STATE (change-device-state "pi1" #t)))

  (check-equal? (new-msg BROKER0 iworld1 MSG1)
                (broker (hash "pi1" #f "monitor1" iworld1)
                        (hash) (list (cons "pi1" iworld1)) '()))

  (check-equal? (new-msg BROKER0 iworld1 MSG2)
                (make-bundle 
                 (broker REGISTRY (hash "pi1" (message-body MSG2)) '() '()) '() '()))

  (check-equal? (new-msg BROKER0 iworld1 MSG3)
                (broker (hash "pi1" iworld1 "monitor1" #f)
                        (hash) '() '()))

  (define BROKER1
    (new-msg
     (new-msg BROKER0 iworld1 MSG1)
     iworld2 MSG3))

  (check-equal? (subscribers (broker-subscriptions BROKER1) "pi1") (list iworld1))

  (define BROKER2
    (new-msg BROKER0 iworld1 (message "foo" MSG-SUBSCRIBE (list "a" "b"))))

  (check-equal? (broker-subscriptions BROKER2) (list
                                                (cons "a" iworld1)
                                                (cons "b" iworld1)))
  
  )
