#lang racket/base

(provide (struct-out device)
         valid-device-state?
         (struct-out change-device-state)
         valid-change-device-state?
         
         (struct-out message)
         MSG-ID
         MSG-SUBSCRIBE         
         MSG-PUBLISH
         MSG-CHANGE-STATE
         valid-message?)

;; device state is a struct
;; - timestamp (number?)
;; - temperature (or/c number? #f) (#f if sending message back to device)
;; - led-state (boolean?)

;; device state is used from the device to tell the broker about its state
;; and it is used to change the state of the LED, then temperature should be #f

(struct device (timestamp temperature led) #:prefab
  #:extra-constructor-name make-device)

;; checks if a given device state is valid
;; device? -> boolean?
(define (valid-device-state? dev)
  (and (device? dev)
       (number? (device-timestamp dev))
       (number? (device-temperature dev))
       (boolean? (device-led dev))))

;; change-device-state is a struct
;; - device id of device that should be changed (string?)
;; - new state (boolean?)
(struct change-device-state (device led) #:prefab
  #:extra-constructor-name make-change-device-state)

;; checks if a given change-device-state is valid
;; change-device-state -> boolean?
(define (valid-change-device-state? chg)
  (and (change-device-state? chg)
       (string? (change-device-state-device chg))
       (boolean? (change-device-state-led chg))))

;; message is a struct
;; - sender (string?)
;; - type (string?, one of "subscribe" "publish-data" "change-stat" "id")
;; - body (one of device? string? string? #f)
;; XXX: LATER: signature 

(struct message (sender type body) #:prefab
  #:extra-constructor-name make-message)

;; Constants for message types
(define MSG-SUBSCRIBE "subscribe")  ;; Subscribe to a topic, body is topic 
(define MSG-ID "id")  ;; Ask for Id, send my id, body is id 
(define MSG-PUBLISH "publish-data") ;; Publish some data, body is device state
(define MSG-CHANGE-STATE "change-state") ;; Change state of a device; body is 

;; checks if a given message is valid
;; message? -> boolean?
(define (valid-message? msg)
  (and (message? msg)
       (string? (message-sender msg))
       (string? (message-type msg))
       (or (equal? (message-type msg) MSG-SUBSCRIBE)
           (equal? (message-type msg) MSG-ID)
           (equal? (message-type msg) MSG-PUBLISH)
           (equal? (message-type msg) MSG-CHANGE-STATE))
       (or (device? (message-body msg))
           (string? (message-body msg))
           (list? (message-body msg))
           (change-device-state? (message-body msg))
           (equal? (message-body msg) #f))))

(module+ test
  (require rackunit)

  (check-true (valid-device-state? (device 12 243 #t)))
  (check-false (valid-device-state? (device 124 #f #t)))
  (check-false (valid-device-state? (device 12 243 5655)))
  (check-false (valid-device-state? (device 12 243 "aaa")))
  
  (check-true (valid-device-state? (make-device 345 10 #true)))
  (check-false (valid-device-state? (make-device 345 10 "abc")))
  (check-true (valid-device-state? (make-device 345 10 #false)))
  (check-false (valid-device-state? (make-device 111 "abc" #t)))
  (check-false (valid-device-state? (make-device "abc" "abc" "def")))

  (check-true (valid-message? (message "foobar" "subscribe" "topic1")))
  (check-true (valid-message? (message "foobar" "publish-data" (device 12 123 #t))))
  (check-false (valid-message? (message 'a #t #t)))

  (check-true (valid-change-device-state? (change-device-state "foo" #f)))
  (check-false (valid-change-device-state? (change-device-state 'fff #f)))
  (check-false (valid-change-device-state? (change-device-state "foo" 'fff)))
  )
