#lang htdp/bsl

(require "common.rkt")
(require 2htdp/image)
(require 2htdp/universe)

;; WorldState is a struct of
;; - timestamp (a number?) 
;; - temperature (a number?)
;; - led-state (a boolean?)

;; Constant for my Id
(define MONITOR-ID "monitor1")
;; Constant for what topic to subscribe to
(define TOPIC "pi01")

;; Constant for server to connect to
(define SERVER LOCALHOST)

;; Constants for monitor window
(define WIDTH 600)
(define HEIGHT 50)

;; Constants for temperature too low and to high
(define TEMP-TOO-LOW 10)
(define TEMP-TOO-HIGH 30)

;; Checks if temperature is too high
;; number? -> boolean?
(define (temperature-too-high? temp)
  (if (> temp TEMP-TOO-HIGH)
      #true
      #false))

(check-expect (temperature-too-high? (- TEMP-TOO-HIGH 10)) #false)
(check-expect (temperature-too-high? (+ TEMP-TOO-HIGH 1)) #true)

;; Checks if temperature is too low
;; number? -> boolean?
(define (temperature-too-low? temp)
  (if (< temp TEMP-TOO-LOW)
      #true
      #false))

(check-expect (temperature-too-low? (- TEMP-TOO-LOW 10)) #true)
(check-expect (temperature-too-low? (+ TEMP-TOO-LOW 1)) #false)

;; Checks if temperatur is okay
;; number? -> boolean?
(define (temperature-ok? temp)
  (and (not (temperature-too-low? temp))
       (not (temperature-too-high? temp))))

;; Renders the monitor
;; WorldState -> WorldState
(define (render-monitor ws)
  (overlay/align "left" "middle"
                 (text (format "Monitoring... [~a]" ws) 25 "blue")
                 (empty-scene WIDTH HEIGHT)))

;; Receives a message
;; Depending on the message different action is to be performed.
;; Types of messages are in "common.rkt"
;; Respond:
;; MSG-ID: Send my subscription 
;; MSG-PUBLISH: Got some data from a device I subscribed to.
;; WorldState sexp? -> HandlerResult (or/c WorldState package?)
(define (receive-message ws msg)
  (if (and (message? msg)
           (valid-message? msg))
      (cond
        [(equal? (message-type msg) MSG-ID)
         (make-package ws (message MONITOR-ID MSG-SUBSCRIBE TOPIC))]
        [(and (equal? (message-type msg) MSG-PUBLISH)
              (device? (message-body msg)))
         (make-package 
          (message-body msg)
          (make-message MONITOR-ID MSG-CHANGE-STATE
                        (make-change-device-state
                         TOPIC
                         (not (temperature-ok? (device-temperature (message-body msg)))))))
         ]
        [else ws])
      ws))

;; Starts big-bang with an initial state
;; WorldState -> WorldState
(define (main start-ws)
  (big-bang start-ws
            (name MONITOR-ID)
            (on-receive receive-message)
            (register SERVER)          
            (to-draw render-monitor)))

(main #true)
