#lang racket/base

(require file/sha1
         racket/system
         racket/port)

(provide get-temperature
         get-light
         set-light
         on-raspberry-pi?)

;; When #f this module is not running on a Raspberry Pi. Therefore
;; all functions return some "test/debug" data. If you want to use
;; it "in production" set this parameter to #t
(define on-raspberry-pi? (make-parameter #f))

;; init gpio 23
(when (on-raspberry-pi?)
  (system "gpio export 23 out"))

;; Gives temperature from LM 75 sensor
;; -> number?
(define (get-temperature)
  (if (on-raspberry-pi?)
      (lm75string->temp
       (substring
        (with-output-to-string
            (lambda () (system "i2cget -y 1 0x48 0x00 w"))) 0 6))
      (random 100)))

;; Global state for a "light":
(define light #f)

;; Gives the status of the LED
;; -> boolean? 
(define (get-light)
  (if (on-raspberry-pi?)
      (equal? "1"
              (substring
               (with-output-to-string
                   (lambda () (system "gpio -g read 23"))) 0 1))
      light))

;; Switches LED light
;; boolean? -> boolean?
(define (set-light new-state)
  (if (on-raspberry-pi?)
      (if new-state
          (system "gpio -g write 23 1")
          (system "gpio -g write 23 0"))
      (set! light new-state))
  (get-light))

;; Convert string of i2cget to temperature
;; string? -> number?
(define (lm75string->temp s)
  (let* ((s-bytes (hex-string->bytes (substring s 2)))
         (high-byte (bytes-ref s-bytes 1))
         (low-byte (bytes-ref s-bytes 0)))
    (if (= (arithmetic-shift high-byte -7) 0)
        (+ high-byte (* (arithmetic-shift low-byte -7) 0.5))
        (* -1 (+ (bitwise-xor high-byte #xff) 1 (* (arithmetic-shift low-byte -7) -0.5))))))

(module+ test
  (require rackunit)
  ;; my checks
  (check-equal? (lm75string->temp "0x8018") 24.5)
  (check-equal? (lm75string->temp "0x80E8") -23.5)
  ;; from spec
  (check-equal? (lm75string->temp "0x007D") 125)
  (check-equal? (lm75string->temp "0x0019") 25)
  (check-equal? (lm75string->temp "0x8000") 0.5)
  (check-equal? (lm75string->temp "0x0000") 0)
  (check-equal? (lm75string->temp "0x80FF") -0.5)
  (check-equal? (lm75string->temp "0x00E7") -25)
  (check-equal? (lm75string->temp "0x00C9") -55)
  )
