#lang racket/base


(require racket/date
         racket/function
         racket/format
         racket/list
         2htdp/image
         2htdp/universe
         "common.rkt")

(define WIDTH 1000)
(define HEIGHT 690)

(define DEVICE-HEIGHT 116)
(define DEVICE-WIDTH 500)

(define RADIUS 15)

(define WALL-ID "wall")

(define SERVER LOCALHOST)

(define DEVICES '("pi01" "pi02" "pi03"))


;; WorldState is a hash of all Pi's device state

(define (wall-state? ws)
  (hash? ws))

;; Renders a single device
;; device? -> image?
(define (render-single-id id dvc)
  (parameterize ([date-display-format 'iso-8601])
    (beside
     (text/font (format "~a ~a:     ~a°C  "
                   (~a (format "[~a]" id)
                       #:max-width 10
                       #:align 'left
                       #:min-width 10)
                   (date->string (seconds->date (device-timestamp dvc)) #t)
                   (~a (~r (device-temperature dvc) #:precision '(= 1))
                       #:min-width 4
                       #:max-width 4
                       #:align 'right)
                   )
           (/ DEVICE-HEIGHT 4) "black" "Courier" 'default 'normal 'normal #f)
     (circle RADIUS "solid" (cond
                              [(equal? (device-led dvc) #t)
                               "red"]
                              [(equal? (device-led dvc) #f)
                               "green"]
                              [else "grey"])))))

;; draws the wall
;; WorldSate -> image?
(define (render-wall ws)
  (cond
    [(> (hash-count ws) 1)
     (apply
      ((curry above/align) "left")
      (hash-map ws (lambda (a b) (render-single-id a b))))]
    [(= (hash-count ws) 1)
     (let ([dvc (first (hash->list ws))])
       (render-single-id (car dvc) (cdr dvc)))]
    [else (empty-scene WIDTH HEIGHT)]))

;; receives a new message from the broker; never gives something back
;; to the broker, only when it is asked to give its id it sends a list
;; of all known ids the wall wants to monitor. 
;; WorlsState sexp? -> HandlerResult
(define (receive-message ws msg)
  (if (and (message? msg)
           (valid-message? msg))
      (cond
        [(equal? (message-type msg) MSG-ID)
         (make-package ws (message WALL-ID MSG-SUBSCRIBE DEVICES))]
        [(and (equal? (message-type msg) MSG-PUBLISH)
              (device? (message-body msg)))
         (hash-set ws (message-sender msg) (message-body msg))]
        [else ws])
      ws))


;; starts big bang with an initial state (list of Pis to monitor), returns otherwise
;; tries to register to SERVER and get messages from the broker.
(define (main start-ws)
  (if (wall-state? start-ws)
      (big-bang start-ws
                (name "The Wall")
                (to-draw render-wall)
                (on-receive receive-message)
                (register SERVER))
      (error "no valid world state given")))

(module+ test
  (require rackunit)

  (define DEV1 (device 1481117128 22.4 #f))
  (define DEV2 (device 1481117355 45.0 #t))

  (define WS (hash "Pi01" DEV1 "Pi02" DEV2))
  )

(module+ main
  (main (hash)))
